<?php
session_start();
require_once "../config/config.php";

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'customer') {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Fetch user data for prefill
$userStmt = $pdo->prepare("SELECT fullname, email FROM users WHERE id = ?");
$userStmt->execute([$user_id]);
$user = $userStmt->fetch();

if (!$user) {
    // User not found; logout for safety
    session_destroy();
    header('Location: login.php');
    exit;
}

require_once "../src/models/Room.php";
require_once "../src/models/Booking.php";

$roomModel = new Room($pdo);
$bookingModel = new Booking($pdo);

$rooms = $roomModel->getAllRooms();
$message = '';
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $room_id = $_POST['room_id'] ?? null;
    $guest_name = trim($_POST['guest_name'] ?? '');
    $guest_email = trim($_POST['guest_email'] ?? '');
    $check_in = $_POST['check_in'] ?? null;
    $check_out = $_POST['check_out'] ?? null;

    // Basic validation
    $errors = [];
    if (!$room_id) $errors[] = "Please select a room.";
    if (!$guest_name) $errors[] = "Guest name is required.";
    if (!$guest_email || !filter_var($guest_email, FILTER_VALIDATE_EMAIL)) $errors[] = "Valid guest email is required.";
    if (!$check_in) $errors[] = "Check-in date is required.";
    if (!$check_out) $errors[] = "Check-out date is required.";
    if ($check_in >= $check_out) $errors[] = "Check-out must be after check-in.";

    if (empty($errors)) {
        try {
            $pdo->beginTransaction();

            $stmt = $pdo->prepare("INSERT INTO bookings (user_id, room_id, guest_name, guest_email, check_in, check_out, status) VALUES (?, ?, ?, ?, ?, ?, 'pending')");
            $success = $stmt->execute([
                $user_id,
                $room_id,
                $guest_name,
                $guest_email,
                $check_in,
                $check_out
            ]);

            if (!$success) {
                throw new Exception('Failed to create booking.');
            }

            $booking_id = $pdo->lastInsertId();

            // Calculate total amount for stay in GHS
            $roomPriceStmt = $pdo->prepare("SELECT price FROM rooms WHERE id = ?");
            $roomPriceStmt->execute([$room_id]);
            $room_price = $roomPriceStmt->fetchColumn();

            $checkInDate = new DateTime($check_in);
            $checkOutDate = new DateTime($check_out);
            $nights = $checkOutDate->diff($checkInDate)->days;

            $amount_ghs = $room_price * $nights;
            $amount_kobo = $amount_ghs * 100; // convert to Kobo for Paystack

            // Initialize Paystack payment
            $secretKey = 'YOUR_PAYSTACK_SECRET_KEY';
            $callback_url = "https://" . $_SERVER['HTTP_HOST'] . "/paystack_callback.php?booking_id=" . $booking_id;

            $curl = curl_init();

            curl_setopt_array($curl, [
                CURLOPT_URL => "https://api.paystack.co/transaction/initialize",
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_CUSTOMREQUEST => "POST",
                CURLOPT_POSTFIELDS => json_encode([
                    'email' => $guest_email,
                    'amount' => (int)$amount_kobo,
                    'callback_url' => $callback_url,
                ]),
                CURLOPT_HTTPHEADER => [
                    "Authorization: Bearer $secretKey",
                    "Content-Type: application/json",
                ],
            ]);

            $response = curl_exec($curl);
            $err = curl_error($curl);
            curl_close($curl);

            if ($err) {
                throw new Exception("Payment initialization error: " . $err);
            }

            $res = json_decode($response);

            if ($res->status && isset($res->data->authorization_url)) {
                $pdo->commit();
                header('Location: ' . $res->data->authorization_url);
                exit;
            } else {
                throw new Exception("Payment initialization failed: " . ($res->message ?? 'Unknown error'));
            }
        } catch (Exception $ex) {
            $pdo->rollBack();
            $message = $ex->getMessage();
        }
    } else {
        $message = implode("<br>", $errors);
    }
}

ob_start();
?>

<style>
    body {
        background: linear-gradient(to right, #e0f7fa, #ffffff);
        font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }
    .booking-container {
        max-width: 600px;
        margin: 40px auto;
        background: #ffffffcc;
        border-radius: 15px;
        box-shadow: 0 8px 16px rgba(0, 150, 193, 0.15);
        padding: 40px;
        backdrop-filter: blur(10px);
        border: 1px solid #b2ebf2;
    }
    .booking-container h2 {
        color: #007c91;
        margin-bottom: 30px;
        font-weight: 700;
        letter-spacing: 1px;
    }
    .btn-ice {
        background-color: #00acc1;
        color: white;
        border-radius: 50px;
        padding: 12px 30px;
        font-weight: 600;
        transition: background-color 0.3s ease;
    }
    .btn-ice:hover {
        background-color: #0097a7;
        color: white;
    }
</style>

<div class="booking-container">
    <h2>Book a Room</h2>

    <?php if ($message): ?>
        <div class="alert <?= $success ? 'alert-success' : 'alert-danger' ?>">
            <?= $message ?>
        </div>
    <?php endif; ?>

    <form method="post" novalidate>
        <div class="mb-4">
            <label for="room_id" class="form-label fw-semibold">Select Room</label>
            <select id="room_id" name="room_id" class="form-select form-select-lg" required>
                <option value="" disabled <?= !isset($_POST['room_id']) ? 'selected' : '' ?>>Choose a room</option>
                <?php foreach ($rooms as $room): ?>
                    <option value="<?= $room['id'] ?>" <?= (isset($_POST['room_id']) && $_POST['room_id'] == $room['id']) ? 'selected' : '' ?>>
                        <?= htmlspecialchars($room['room_number']) ?> - <?= htmlspecialchars($room['type']) ?> (GHS <?= number_format($room['price'], 2) ?>)
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="mb-4">
            <label for="guest_name" class="form-label fw-semibold">Full Name</label>
            <input type="text" id="guest_name" name="guest_name" class="form-control form-control-lg" placeholder="Your name"
                value="<?= htmlspecialchars($_POST['guest_name'] ?? $user['fullname']) ?>" required />
        </div>

        <div class="mb-4">
            <label for="guest_email" class="form-label fw-semibold">Email Address</label>
            <input type="email" id="guest_email" name="guest_email" class="form-control form-control-lg" placeholder="you@example.com"
                value="<?= htmlspecialchars($_POST['guest_email'] ?? $user['email']) ?>" required />
        </div>

        <div class="row mb-4">
            <div class="col-md-6">
                <label for="check_in" class="form-label fw-semibold">Check-in Date</label>
                <input type="date" id="check_in" name="check_in" class="form-control form-control-lg" value="<?= htmlspecialchars($_POST['check_in'] ?? '') ?>" required />
            </div>
            <div class="col-md-6">
                <label for="check_out" class="form-label fw-semibold">Check-out Date</label>
                <input type="date" id="check_out" name="check_out" class="form-control form-control-lg" value="<?= htmlspecialchars($_POST['check_out'] ?? '') ?>" required />
            </div>
        </div>

        <button type="submit" class="btn btn-ice w-100">Book Now</button>
    </form>
</div>

<?php
$content = ob_get_clean();
$page_title = "Book a Room";
require "../src/views/layouts/public_layout.php";
?>
