<?php
session_start();
require_once __DIR__ . '/../src/middleware/auth.php';
requireAdmin();
require_once "../config/config.php";

if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header('Location: manage_rooms.php');
    exit;
}

$room_id = (int)$_GET['id'];
$message = '';

$stmt = $pdo->prepare("SELECT * FROM rooms WHERE id = ?");
$stmt->execute([$room_id]);
$room = $stmt->fetch();

if (!$room) {
    header('Location: manage_rooms.php');
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $room_number = trim($_POST['room_number']);
    $type = trim($_POST['type']);
    $description = trim($_POST['description']);
    $price = floatval($_POST['price']);
    $status = $_POST['status'];
    $max_occupancy = intval($_POST['max_occupancy'] ?? 1);
    $amenities = trim($_POST['amenities'] ?? '');
    $featured = isset($_POST['featured']) ? 1 : 0;

    $errors = [];

    if (!$room_number) {
        $errors[] = 'Room number is required.';
    }
    if (!$type) {
        $errors[] = 'Room type is required.';
    }
    if ($price <= 0) {
        $errors[] = 'Price must be greater than zero.';
    }
    if (!in_array($status, ['available', 'occupied', 'maintenance'])) {
        $errors[] = 'Invalid room status.';
    }
    if ($max_occupancy <= 0) {
        $errors[] = 'Max occupancy must be at least 1.';
    }

    if (empty($errors)) {
        $stmt = $pdo->prepare("UPDATE rooms SET room_number = ?, type = ?, description = ?, price = ?, status = ?, max_occupancy = ?, amenities = ?, featured = ? WHERE id = ?");
        $updated = $stmt->execute([$room_number, $type, $description, $price, $status, $max_occupancy, $amenities, $featured, $room_id]);
        if ($updated) {
            $message = "Room updated successfully!";
            // Refresh local copy for form display
            $room = array_merge($room, [
                'room_number' => $room_number,
                'type' => $type,
                'description' => $description,
                'price' => $price,
                'status' => $status,
                'max_occupancy' => $max_occupancy,
                'amenities' => $amenities,
                'featured' => $featured
            ]);
        } else {
            $message = "Failed to update room.";
        }
    } else {
        $message = implode('<br>', $errors);
    }
}

ob_start();
?>

<div class="container-fluid">
    <h2>Edit Room #<?= $room_id ?></h2>

    <?php if ($message): ?>
        <div class="alert <?= strpos(strtolower($message), 'success') !== false ? 'alert-success' : 'alert-danger' ?>">
            <?= $message ?>
        </div>
    <?php endif; ?>

    <form method="post" novalidate>
        <div class="mb-3">
            <label for="room_number" class="form-label">Room Number</label>
            <input type="text" class="form-control" id="room_number" name="room_number" required value="<?= htmlspecialchars($room['room_number'] ?? '') ?>" />
        </div>
        <div class="mb-3">
            <label for="type" class="form-label">Room Type</label>
            <input type="text" class="form-control" id="type" name="type" required value="<?= htmlspecialchars($room['type'] ?? '') ?>" />
        </div>
        <div class="mb-3">
            <label for="description" class="form-label">Description</label>
            <textarea class="form-control" id="description" name="description" rows="3"><?= htmlspecialchars($room['description'] ?? '') ?></textarea>
        </div>
        <div class="mb-3">
            <label for="price" class="form-label">Price (GHS)</label>
            <input type="number" step="0.01" min="0" class="form-control" id="price" name="price" required value="<?= htmlspecialchars($room['price'] ?? '') ?>" />
        </div>
        <div class="mb-3">
            <label for="max_occupancy" class="form-label">Max Occupancy</label>
            <input type="number" min="1" class="form-control" id="max_occupancy" name="max_occupancy" required value="<?= htmlspecialchars($room['max_occupancy'] ?? '1') ?>" />
        </div>
        <div class="mb-3">
            <label for="amenities" class="form-label">Amenities (comma-separated)</label>
            <input type="text" class="form-control" id="amenities" name="amenities" value="<?= htmlspecialchars($room['amenities'] ?? '') ?>" />
        </div>
        <div class="mb-3">
            <label for="status" class="form-label">Status</label>
            <select id="status" name="status" class="form-select" required>
                <?php
                    $statuses = ['available' => 'Available', 'occupied' => 'Occupied', 'maintenance' => 'Maintenance'];
                    $selected_status = $room['status'] ?? 'available';
                    foreach ($statuses as $value => $label):
                ?>
                    <option value="<?= $value ?>" <?= $selected_status === $value ? 'selected' : '' ?>><?= $label ?></option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="mb-3 form-check">
            <input type="checkbox" class="form-check-input" id="featured" name="featured" <?= !empty($room['featured']) ? 'checked' : '' ?>>
            <label class="form-check-label" for="featured">Feature this room</label>
        </div>

        <button type="submit" class="btn btn-primary">Update Room</button>
        <a href="manage_rooms.php" class="btn btn-secondary ms-2">Back</a>
    </form>
</div>

<?php
$content = ob_get_clean();
$page_title = "Edit Room";
require "../src/views/layouts/admin_layout.php";
