<?php
session_start();
require_once __DIR__ . '/../src/middleware/auth.php';
requireAdmin();
require_once "../config/config.php";
require_once "../src/models/Booking.php";
require_once "../src/models/Room.php";

$bookingModel = new Booking($pdo);
$roomModel = new Room($pdo);

// Summary stats
$totalRooms = count($roomModel->getAllRooms());
$totalBookingsResult = $pdo->query("SELECT COUNT(*) FROM bookings");
$totalBookings = $totalBookingsResult->fetchColumn();

$today = date('Y-m-d');
// Sum prices from rooms for bookings created today
$revenueStmt = $pdo->prepare("
    SELECT IFNULL(SUM(r.price), 0) 
    FROM bookings b 
    JOIN rooms r ON b.room_id = r.id 
    WHERE DATE(b.created_at) = ?
");
$revenueStmt->execute([$today]);
$revenueToday = $revenueStmt->fetchColumn();

$statusCounts = [];
$statusQuery = $pdo->query("SELECT status, COUNT(*) as count FROM bookings GROUP BY status");
while (($row = $statusQuery->fetch()) !== false) {
    $statusCounts[$row['status']] = $row['count'];
}

// Recent bookings (latest 5)
$recentBookingsStmt = $pdo->query("
    SELECT b.*, r.room_number, r.type
    FROM bookings b
    JOIN rooms r ON b.room_id = r.id
    ORDER BY b.created_at DESC LIMIT 5
");
$recentBookings = $recentBookingsStmt->fetchAll();

// Notifications count
$notifCountStmt = $pdo->query("SELECT COUNT(*) FROM admin_notifications WHERE is_read = FALSE");
$notificationsCount = $notifCountStmt->fetchColumn();

// Top booked room types (top 5)
$topRoomTypesStmt = $pdo->query("
    SELECT r.type, COUNT(*) AS booking_count 
    FROM bookings b 
    JOIN rooms r ON b.room_id = r.id 
    GROUP BY r.type 
    ORDER BY booking_count DESC LIMIT 5
");
$topRoomTypes = $topRoomTypesStmt->fetchAll();

// Rooms under maintenance
$maintenanceRoomsStmt = $pdo->query("SELECT COUNT(*) FROM rooms WHERE status = 'maintenance'");
$maintenanceRoomsCount = $maintenanceRoomsStmt->fetchColumn();

// Occupied rooms
$occupiedRoomsStmt = $pdo->query("SELECT COUNT(*) FROM rooms WHERE status = 'occupied'");
$occupiedRoomsCount = $occupiedRoomsStmt->fetchColumn();

// Monthly revenue & occupancy chart data for past 12 months
$startDate = date('Y-m-d', strtotime('-11 months', strtotime(date('Y-m-01'))));
$monthlyDataStmt = $pdo->prepare("
  SELECT DATE_FORMAT(b.created_at, '%Y-%m') AS month, 
         IFNULL(SUM(r.price), 0) AS revenue
  FROM bookings b 
  JOIN rooms r ON b.room_id = r.id 
  WHERE b.created_at >= ?
  GROUP BY month 
  ORDER BY month ASC
");
$monthlyDataStmt->execute([$startDate]);
$monthlyData = $monthlyDataStmt->fetchAll();

$months = [];
$revenues = [];
foreach ($monthlyData as $data) {
    $months[] = $data['month'];
    $revenues[] = (float)$data['revenue'];
}

ob_start();
?>

<style>
    .dashboard-container {
        max-width: 1200px;
        margin: 0 auto;
    }
    .card-deck {
        display: flex;
        flex-wrap: wrap;
        justify-content: center;
        gap: 20px;
    }
    .dashboard-card {
        flex: 0 0 31%;
        max-width: 31%;
        height: 230px;
        margin-bottom: 1.5rem;
        display: flex;
        flex-direction: column;
        justify-content: center;
        text-align: center;
    }
    .dashboard-card .card {
        height: 100%;
        width: 100%;
        padding: 1rem;
    }
    .dashboard-card .card-title {
        font-size: 1.2rem;
        font-weight: 600;
        margin-bottom: 0.7rem;
    }
    .dashboard-card .display-5 {
        font-size: 2.2rem;
        font-weight: bold;
    }
    .booking-action-btn {
        padding: 0.3rem 0.6rem;
        font-size: 0.75rem;
    }
    /* Responsive for very small screens */
    @media (max-width: 575.98px) {
        .dashboard-card {
            flex: 0 0 100%;
            max-width: 100%;
            height: auto;
        }
        .dashboard-card .card {
            height: auto;
        }
    }
    /* Small devices: 2 cards per row */
    @media (min-width: 576px) and (max-width: 767.98px) {
        .dashboard-card {
            flex: 0 0 48%;
            max-width: 48%;
            height: 220px;
        }
        .dashboard-card .card {
            height: 100%;
        }
    }
    /* Medium+ devices: 3 cards per row */
    @media (min-width: 768px) {
        .dashboard-card {
            flex: 0 0 31%;
            max-width: 31%;
            height: 230px;
        }
        .dashboard-card .card {
            height: 100%;
        }
    }
    .table-responsive {
        overflow-x: auto;
    }
    table td, table th {
        white-space: nowrap;
    }
</style>

<div class="container-fluid dashboard-container">
    <h1 class="mb-4">Dashboard</h1>

    <div class="card-deck">
        <div class="dashboard-card">
            <div class="card shadow-sm text-white bg-primary">
                <div class="card-body">
                    <h5 class="card-title">Total Rooms</h5>
                    <p class="display-5"><?= $totalRooms ?></p>
                </div>
            </div>
        </div>
        <div class="dashboard-card">
            <div class="card shadow-sm text-white bg-info">
                <div class="card-body">
                    <h5 class="card-title">Total Bookings</h5>
                    <p class="display-5"><?= $totalBookings ?></p>
                </div>
            </div>
        </div>
        <div class="dashboard-card">
            <div class="card shadow-sm text-white bg-success">
                <div class="card-body">
                    <h5 class="card-title">Today's Revenue (GHS)</h5>
                    <p class="display-5"><?= number_format($revenueToday, 2) ?></p>
                </div>
            </div>
        </div>
        <div class="dashboard-card">
            <div class="card shadow-sm text-white bg-warning">
                <div class="card-body">
                    <h5 class="card-title">Pending Bookings</h5>
                    <p class="display-5"><?= $statusCounts['pending'] ?? 0 ?></p>
                </div>
            </div>
        </div>
        <div class="dashboard-card">
            <div class="card shadow-sm text-white bg-secondary">
                <div class="card-body">
                    <h5 class="card-title">Unread Notifications</h5>
                    <p class="display-5"><?= $notificationsCount ?></p>
                </div>
            </div>
        </div>
        <div class="dashboard-card">
            <div class="card shadow-sm text-white bg-danger">
                <div class="card-body">
                    <h5 class="card-title">Maintenance Rooms</h5>
                    <p class="display-5"><?= $maintenanceRoomsCount ?></p>
                </div>
            </div>
        </div>
    </div>

    <h2 class="mt-5 mb-3">Recent Bookings</h2>
    <div class="table-responsive">
        <table class="table table-striped shadow-sm mb-4">
            <thead>
                <tr>
                    <th>Room</th>
                    <th>Guest</th>
                    <th>Check-in</th>
                    <th>Check-out</th>
                    <th>Status</th>
                    <th>Action</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($recentBookings as $booking): ?>
                <tr>
                    <td><?= htmlspecialchars($booking['room_number'] . ' - ' . $booking['type']) ?></td>
                    <td><?= htmlspecialchars($booking['guest_name']) ?></td>
                    <td><?= htmlspecialchars($booking['check_in']) ?></td>
                    <td><?= htmlspecialchars($booking['check_out']) ?></td>
                    <td><?= ucfirst(htmlspecialchars($booking['status'])) ?></td>
                    <td>
                        <?php if ($booking['status'] === 'pending'): ?>
                        <form method="post" action="booking_action.php" class="d-inline">
                            <input type="hidden" name="booking_id" value="<?= $booking['id'] ?>" />
                            <button type="submit" name="action" value="confirm" class="btn btn-success booking-action-btn">Confirm</button>
                        </form>
                        <form method="post" action="booking_action.php" class="d-inline ms-1">
                            <input type="hidden" name="booking_id" value="<?= $booking['id'] ?>" />
                            <button type="submit" name="action" value="cancel" class="btn btn-danger booking-action-btn">Cancel</button>
                        </form>
                        <?php else: ?>
                        <em>No actions</em>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <h2>Top Booked Room Types</h2>
    <ul class="list-group mb-4" style="max-width: 400px;">
        <?php foreach ($topRoomTypes as $type): ?>
        <li class="list-group-item d-flex justify-content-between align-items-center">
            <?= htmlspecialchars($type['type']) ?>
            <span class="badge bg-primary rounded-pill"><?= $type['booking_count'] ?></span>
        </li>
        <?php endforeach; ?>
    </ul>

    <h2 class="mb-3">Monthly Revenue</h2>
    <canvas id="dashboardChart" style="max-width: 900px;"></canvas>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
    const ctx = document.getElementById('dashboardChart').getContext('2d');

    const months = <?= json_encode($months) ?>;
    const revenues = <?= json_encode($revenues) ?>;

    const dashboardChart = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: months,
            datasets: [{
                label: 'Revenue (GHS)',
                data: revenues,
                backgroundColor: 'rgba(54, 162, 235, 0.7)',
                yAxisID: 'y',
            }]
        },
        options: {
            interaction: {
                mode: 'nearest',
                intersect: true
            },
            scales: {
                y: {
                    beginAtZero: true,
                    position: 'left',
                    title: {
                        display: true,
                        text: 'Revenue (GHS)'
                    }
                }
            }
        }
    });
</script>

<?php
$content = ob_get_clean();
$page_title = "Admin Dashboard";
require "../src/views/layouts/admin_layout.php";
