<?php
session_start();
require_once __DIR__ . '/../src/middleware/auth.php';
requireAdmin();
require_once "../config/config.php";

$message = '';
$imageFileName = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $room_number = trim($_POST['room_number']);
    $type = trim($_POST['type']);
    $description = trim($_POST['description']);
    $price = floatval($_POST['price']);
    $status = $_POST['status'];
    $max_occupancy = intval($_POST['max_occupancy'] ?? 1);
    $amenities = trim($_POST['amenities'] ?? '');
    $featured = isset($_POST['featured']) ? 1 : 0;

    $errors = [];

    if (!$room_number) {
        $errors[] = 'Room number is required.';
    }
    if (!$type) {
        $errors[] = 'Room type is required.';
    }
    if ($price <= 0) {
        $errors[] = 'Price must be greater than zero.';
    }
    if (!in_array($status, ['available', 'occupied', 'maintenance'])) {
        $errors[] = 'Invalid room status.';
    }
    if ($max_occupancy <= 0) {
        $errors[] = 'Max occupancy must be at least 1.';
    }

    if (isset($_FILES['image']) && $_FILES['image']['error'] !== UPLOAD_ERR_NO_FILE) {
        $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif'];
        $uploadDir = __DIR__ . '/../uploads/rooms/';

        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }

        $tmpName = $_FILES['image']['tmp_name'];
        $originalName = basename($_FILES['image']['name']);
        $ext = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));

        if (!in_array($ext, $allowedExtensions)) {
            $errors[] = "Room image must be one of: " . implode(', ', $allowedExtensions);
        } else {
            $imageFileName = uniqid('room_') . '.' . $ext;
            $destination = $uploadDir . $imageFileName;

            if (!move_uploaded_file($tmpName, $destination)) {
                $errors[] = "Failed to upload room image.";
            }
        }
    }

    if (empty($errors)) {
        $stmt = $pdo->prepare("
            INSERT INTO rooms 
            (room_number, type, description, price, status, image, max_occupancy, amenities, featured)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");

        if ($stmt->execute([$room_number, $type, $description, $price, $status, $imageFileName, $max_occupancy, $amenities, $featured])) {
            $message = "Room added successfully!";
            $_POST = [];
            $imageFileName = '';
        } else {
            $message = "Failed to add room.";
        }
    } else {
        $message = implode('<br>', $errors);
    }
}

ob_start();
?>

<div class="container-fluid">
    <h2>Add New Room</h2>

    <?php if ($message): ?>
        <div class="alert <?= strpos(strtolower($message), 'success') !== false ? 'alert-success' : 'alert-danger' ?>">
            <?= $message ?>
        </div>
    <?php endif; ?>

    <form method="post" enctype="multipart/form-data" novalidate>
        <div class="mb-3">
            <label for="room_number" class="form-label">Room Number</label>
            <input type="text" class="form-control" id="room_number" name="room_number" required value="<?= htmlspecialchars($_POST['room_number'] ?? '') ?>" />
        </div>

        <div class="mb-3">
            <label for="type" class="form-label">Room Type</label>
            <input type="text" class="form-control" id="type" name="type" required value="<?= htmlspecialchars($_POST['type'] ?? '') ?>" />
        </div>

        <div class="mb-3">
            <label for="description" class="form-label">Description</label>
            <textarea class="form-control" id="description" name="description" rows="3"><?= htmlspecialchars($_POST['description'] ?? '') ?></textarea>
        </div>

        <div class="mb-3">
            <label for="price" class="form-label">Price (GHS)</label>
            <input type="number" step="0.01" min="0" class="form-control" id="price" name="price" required value="<?= htmlspecialchars($_POST['price'] ?? '') ?>" />
        </div>

        <div class="mb-3">
            <label for="max_occupancy" class="form-label">Max Occupancy</label>
            <input type="number" min="1" class="form-control" id="max_occupancy" name="max_occupancy" required value="<?= htmlspecialchars($_POST['max_occupancy'] ?? '1') ?>" />
        </div>

        <div class="mb-3">
            <label for="amenities" class="form-label">Amenities (comma-separated)</label>
            <input type="text" class="form-control" id="amenities" name="amenities" value="<?= htmlspecialchars($_POST['amenities'] ?? '') ?>" />
        </div>

        <div class="mb-3">
            <label for="status" class="form-label">Status</label>
            <select id="status" name="status" class="form-select" required>
                <?php
                    $statuses = ['available' => 'Available', 'occupied' => 'Occupied', 'maintenance' => 'Maintenance'];
                    $selected_status = $_POST['status'] ?? 'available';
                    foreach ($statuses as $value => $label):
                ?>
                    <option value="<?= $value ?>" <?= $selected_status === $value ? 'selected' : '' ?>><?= $label ?></option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="mb-3 form-check">
            <input type="checkbox" class="form-check-input" id="featured" name="featured" <?= isset($_POST['featured']) ? 'checked' : '' ?>>
            <label class="form-check-label" for="featured">Feature this room</label>
        </div>

        <div class="mb-3">
            <label for="image" class="form-label">Room Image</label>
            <input type="file" class="form-control" id="image" name="image" accept=".jpg,.jpeg,.png,.gif" />
            <?php if ($imageFileName): ?>
                <img src="../uploads/rooms/<?= htmlspecialchars($imageFileName) ?>" alt="Room Image" style="max-width: 200px; margin-top: 10px;" />
            <?php endif; ?>
        </div>

        <button type="submit" class="btn btn-primary">Add Room</button>
        <a href="manage_rooms.php" class="btn btn-secondary ms-2">Back</a>
    </form>
</div>

<?php
$content = ob_get_clean();
$page_title = "Add Room";
require "../src/views/layouts/admin_layout.php";
